const stocksDatatable = document.querySelector('#stocks-datatable');
const roles = JSON.parse(stocksDatatable.dataset.roles);
const depots = JSON.parse(stocksDatatable.dataset.depots);
let stocks = JSON.parse(stocksDatatable.dataset.stocks);
const searchInput = document.getElementById('datatable-search-input');
const spinner = document.querySelector('#spinner');
const depotSelectContent = document.querySelector('#depot')?.innerHTML;
const procurementModalElt = document.querySelector('#procurement-modal');
const destockingModalElt = document.querySelector('#destocking-modal');
const procurementModal = new mdb.Modal(procurementModalElt);
const destockingModal = new mdb.Modal(destockingModalElt);
const baseUrl = localStorage.getItem('baseUrl');

const loadFromServer = async (query) => {
    const q = query.trim().toLowerCase();
    const url = new URL('fichestock', baseUrl);
    url.searchParams.append('search', encodeURI(q));
    url.searchParams.append('limit', 10);

    let response = await fetch(url, {
        headers: new Headers({'X-Requested-With': 'XMLHttpRequest'})
    });

    let data = await response.json();

    return data.data;
};

const setStocks = (stocks) => {
    return stocks.map(row => {
        let alertType;
        const qtyToOrder = row.quantity <= row.warningStockLevel ? row.idealStockLevel - row.quantity : 0;
        let result;

        if (row.quantity === 0) {
            alertType = 'danger';
        } else if (row.quantity <= row.warningStockLevel) {
            alertType = 'warning';
        } else if (row.quantity >= row.idealStockLevel) {
            alertType = 'success';
        } else {
            alertType = 'info';
        }

        result = {
            product: row.product,
            quantity: `${row.quantity} ${(typeof row.uomSymbol === 'string' || typeof row.uomName === 'string') && row.quantity > 1 ? (row.uomSymbol ?? row.uomName) + 's' : ''}`,
            warningStockLevel: `<input type="number" class="form-control warning-stock-level" value="${row.warningStockLevel}" min="0" data-depot-detail-id=${row.id}>`,
            idealStockLevel: `<input type="number" class="form-control ideal-stock-level" value="${row.idealStockLevel}" min="0" data-depot-detail-id=${row.id}>`,
            status: `<span class="badge rounded-pill bg-${alertType} p-2"> </span>`,
            qtyToOrder: (qtyToOrder > 0 ? qtyToOrder : 0)
        };

        if (roles.indexOf('ROLE_GESTSTOCK') !== -1) {
            Object.assign(result, { depot: row.depot });
        }

        Object.assign(result, { action: `<span class="mr-3" data-depot-detail-id="${row.id}" onclick="showProcurementModal(event)" style="cursor: pointer;"><i class="fas fa-plus"></i></span><span data-depot-detail-id="${row.id}" onclick="showDestockingModal(event)" style="cursor: pointer;"><i class="fas fa-minus"></i></span>` });

        return result;
    });
};

const columns = [
    { label: 'Produit', field: 'product' },
    { label: 'Stock disponible', field: 'quantity' },
    { label: 'Avertissement', field: 'warningStockLevel' },
    { label: 'Niveau idéal', field: 'idealStockLevel' },
    { label: 'Statut', field: 'status' },
    { label: 'Qté à commander', field: 'qtyToOrder' }
];

if (roles.indexOf('ROLE_GESTSTOCK') !== -1) {
    columns.push({ label: 'Dépôt', field: 'depot' });
}

columns.push({ label: 'Action', field: 'action' });

const stocksTableInstance = new Datatable(stocksDatatable, {
    columns: columns,
    rows: setStocks(stocks)
});

const loadData = (value) => {
    const filteredStocks = stocks.filter(stock => stock.product.trim().toLowerCase().indexOf(value) > -1);
    const updateTable = (data) => {
        // stocksTableInstance.update({ rows: setStocks(data) }, { loading: false });
        stocksTableInstance.update({ rows: setStocks(data) });
    }
    loadFromServer(value).then((data) => {
        if (filteredStocks.length < data.length) {
            updateTable(data);
        }
        stocks = data;
    });
    updateTable(filteredStocks);
};

function Datatable(datatable, initialContent) {
    this.table = document.createElement('table');
    this.thead = document.createElement('thead');
    this.tbody = document.createElement('tbody');

    this.init = () => {
        this.update(initialContent);
        this.table.classList = 'table table-stripped';
        this.table.appendChild(this.thead);
        this.table.appendChild(this.tbody);
        datatable.appendChild(this.table);
    };

    this.update = (content) => {
        const columns = content?.columns;
        const rows = content?.rows;

        if (typeof columns !== 'undefined' && columns.length > 0) {
            const tr = document.createElement('tr');

            columns.forEach(column => {
                const th = document.createElement('th');
                th.textContent = column.label;
                th.dataset.field = column.field;
                tr.appendChild(th);
            });

            this.thead.innerHTML = '';
            this.thead.appendChild(tr);
            this.columns = columns;
        }

        if (typeof rows !== 'undefined' && rows.length > 0) {
            this.tbody.innerHTML = '';

            rows.forEach((row, index) => {
                const tr = document.createElement('tr');

                for (const key in row) {
                    if (typeof this.columns !== 'undefined' && typeof this.columns.find(column => column.field === key) !== 'undefined') {
                        const td = document.createElement('td');
                        td.innerHTML = row[key];
                        td.dataset.field = key;
                        tr.appendChild(td);
                    }
                }

                this.tbody.appendChild(tr);
            });

            this.tbody.querySelectorAll('.warning-stock-level, .ideal-stock-level').forEach((inputField, idx) => {
                inputField.addEventListener('blur', (e) => {
                    const input = e.currentTarget;
                    const depotDetailId = Number.parseInt(input.dataset.depotDetailId, 10);
                    const inputValue = Number.parseFloat(input.value);
                    // const rowIndex = Number.parseInt(input.parentNode.parentNode.id.split('-')[1], 10);
                    const url = new URL(`fichestock/${depotDetailId}`, baseUrl);

                    fetch(url, {
                        method: 'POST',
                        body: JSON.stringify({[input.parentNode.dataset.field]: inputValue})
                    }).then(response => {
                        if (response.ok) {
                            response.text().then((resp) => {
                                const currentRow = stocks.find(row => row.id === depotDetailId);
                                if (typeof currentRow !== 'undefined') {
                                    Object.assign(currentRow, { [idx % 2 === 0 ? 'warningStockLevel' : 'idealStockLevel']: inputValue });
                                    this.update({ rows: setStocks(stocks) });
                                }
                            });
                        } else {
                            response.text().then(resp => {
                                showAlert(resp, 'danger', false);
                            });
                        }
                    }).catch(() => {
                        showAlert('Vérifiez votre connexion. Si le problème persiste, veuillez contacter l\'administrateur du système.', 'danger', false);
                    });
                });
            });

            this.rows = rows;
        }
    };

    this.init();
}

window.showProcurementModal = function(event) {
    const depotDetailId = event.currentTarget.dataset.depotDetailId;
    const stockRow = stocks.find(stock => stock.id === Number.parseInt(depotDetailId, 10));

    procurementModalElt.dataset.depotDetailId = depotDetailId;

    procurementModalElt.addEventListener('shown.bs.modal', () => {
        const actionButton = event.currentTarget;
        const productElt = procurementModalElt.querySelector('#service');
        const depotElt = procurementModalElt.querySelector('#depot');

        procurementModalElt.querySelectorAll('input').forEach((input) => {
            if (input instanceof HTMLInputElement) {
                input.value = '';
            }
        });

        if (typeof stockRow === 'undefined') {
            productElt.value = '';
            productElt.removeAttribute('disabled');
            procurementModalElt.querySelector('#quantity').value = '';

            if (depotElt !== null) {
                depotElt.innerHTML = depotSelectContent;
                depotElt.removeAttribute('disabled');
            }
        } else {
            const currentDepot = depots.find(d => stockRow.depotId === d.id && stockRow.depot === d.name);
            productElt.value = stockRow.product;
            productElt.setAttribute('disabled', 'disabled');
            procurementModalElt.querySelector('#quantity').value = stockRow.warningStockLevel >= stockRow.quantity ? stockRow.idealStockLevel - stockRow.quantity : '';

            if (depotElt !== null) {
                depotElt.innerHTML = typeof currentDepot !== 'undefined' ? `<option value="${currentDepot.typeDepot}-${currentDepot.id}">${currentDepot.name}</option>` : depotSelectContent;
                if (typeof currentDepot !== 'undefined') {
                    depotElt.setAttribute('disabled', 'disabled');
                } else {
                    depotElt.removeAttribute('disabled');
                }
            }
        }
    });

    procurementModal.show();
};

window.showDestockingModal = function(event) {
    const depotDetailId = Number.parseInt(event.currentTarget.dataset.depotDetailId, 10);
    const stockRow = stocks.find(stock => stock.id === depotDetailId);

    destockingModalElt.dataset.depotDetailId = depotDetailId;
    destockingModalElt.dataset.quantity = stockRow.quantity;
    destockingModalElt.dataset.depotId = stockRow.depotId;
    destockingModalElt.dataset.depot = stockRow.depot;

    destockingModalElt.addEventListener('shown.bs.modal', () => {
        destockingModalElt.querySelector('#nbUnites').value = stockRow.quantity;
        destockingModalElt.querySelector('#stock').value = stockRow.product;
        destockingModalElt.querySelector('#depot-destocking').value = destockingModalElt.dataset.depot;
    });

    destockingModal.show();
};

// document.querySelector('#action-button').addEventListener('click', (event) => {
//     showProcurementModal(event);
// });

searchInput.addEventListener('input', (event) => {
    const value = event.currentTarget.value;
    if (value.length > 1) {
        // stocksTableInstance.update(null, { loading: true });
        stocksTableInstance.update(null);
        loadData(value);
    }
});

document.querySelector('#save-procurement').addEventListener('click', (event) => {
    event.preventDefault();
    const payload = {};
    spinner.removeAttribute('hidden');

    document.querySelectorAll('#procurement-form input, #procurement-form select').forEach((input) => {
        if (input.name !== '') {
            Object.assign(payload, { [input.name]: input.value });
        }
    });

    fetch(`${localStorage.getItem('baseUrl')}/depots/approvisionnement`, {
        method: 'POST',
        headers: new Headers({'X-Requested-With': 'XMLHttpRequest'}),
        body: JSON.stringify(payload)
    }).then((response) => {
        if (response.ok) {
            response.text().then((respBody) => {
                const { message, data } = JSON.parse(respBody);
                const currentRow = stocks.find(row => row.id === Number.parseInt(procurementModalElt.dataset.depotDetailId, 10));

                if (typeof currentRow !== 'undefined') {
                    const newQty = currentRow.quantity + data['quantity'];
                    Object.assign(currentRow, { quantity: newQty });
                    stocksTableInstance.update({ rows: setStocks(stocks) });
                // } else {
                //     if (stocks.length > 0) {
                //         // stocks.unshift();
                //     }
                }

                spinner.setAttribute('hidden', 'hidden');
                // procurementModal.hide();
                showAlert(message, 'success', false);
            });
        } else {
            response.text().then((respBody) => {
                const message = JSON.parse(respBody).message;
                spinner.setAttribute('hidden', 'hidden');
                showAlert(message, 'danger', false);
            });
        }
    });
});

let reasons = [];
const userId = localStorage.getItem('userId');
const token = localStorage.getItem('token');
const reasonsFilter = async (query) => {
    const url = `${localStorage.getItem('baseUrl')}/api/destocking_reasons.json?name=${encodeURI(query.trim())}`;
    const response = await fetch(url, {
        headers: new Headers({'Authorization': 'Bearer ' + token})
    });
    const data = await response.json();
    return data;
};
const reasonAutocomplete = document.querySelector('#reason');
const reasonDropdownMenu = document.querySelector('#reason-dropdown');

window.selectReason = function(event) {
    const currentTarget = event.currentTarget;
    reasonAutocomplete.value = currentTarget.textContent;
    reasonAutocomplete.dataset.code = currentTarget.dataset.code;
    reasonDropdownMenu.style.display = 'none';
};

reasonAutocomplete.addEventListener('input', (e) => {
    const searchQuery = e.currentTarget.value;
    if (searchQuery.length > 1) {
        const reasonSpinner = document.querySelector('#reason-spinner');
        reasonSpinner.removeAttribute('hidden');
        reasonsFilter(searchQuery).then(data => {
            reasonSpinner.setAttribute('hidden', 'hidden');
            reasonDropdownMenu.style.display = 'block';
            if (data.length > 0) {
                let listItems = '';
                data.forEach(item => {
                    listItems += `<li class="dropdown-item" data-code="${item.code}" onclick="selectReason(event)">${item.name}</li>`;
                });
                reasonDropdownMenu.innerHTML = listItems;
            } else {
                reasonDropdownMenu.innerHTML = '';
            }
        });
    } else {
        reasonDropdownMenu.style.display = 'none';
    }
});

document.getElementById('saveBtn').addEventListener('click', () => {
    const destockedQty = Number.parseFloat(document.querySelector('#nbUnites').value);
    const oldQty = Number.parseFloat(destockingModalElt.dataset.quantity);
    const newQty = oldQty - destockedQty;
    
    if (newQty < 0) {
        showAlert('Vous ne pouvez pas destocker une quantité supérieure à celle disponible dans le stock !', 'danger', false);
    } else {
        const stockHeaders = new Headers({ 'Content-Type': 'application/json', 'Authorization': `Bearer ${token}` });
        const stockUrl = `${localStorage.getItem('baseUrl')}/api/${roles.indexOf('ROLE_GESTSTOCK') === -1 ? 'point_of_sale_details' : 'depot_details'}/${destockingModalElt.dataset.depotDetailId}`;

        spinner.removeAttribute('hidden');

        fetch(stockUrl, {
            method: 'GET',
            headers: new Headers({ 'Authorization': `Bearer ${token}`, 'Accept': 'application/json' })
        }).then(response => {
            if (response.ok) {
                response.text().then((dd) => {
                    const depotDetail = JSON.parse(dd);
                    Object.assign(depotDetail, { quantity: newQty });

                    fetch(stockUrl, {
                        method: 'PUT',
                        body: JSON.stringify(depotDetail),
                        headers: stockHeaders
                    }).then(resp => {
                        if (resp.ok) {
                            resp.text().then((respBody) => {
                                const historyHeaders = new Headers({ 'Content-Type': 'application/json', 'Authorization': `Bearer ${token}`, 'Accept': 'application/json' });
                                const data = JSON.parse(respBody);
                                const history = {
                                    quantity: 0 - destockedQty,
                                    [roles.indexOf('ROLE_GESTSTOCK') === -1 ? 'pos' : 'depot']: data[roles.indexOf('ROLE_GESTSTOCK') === -1 ? 'pointOfSale' : 'depot'],
                                    service: data.service,
                                    stockManager: '/api/users/' + userId,
                                    motifDestockage: reasonAutocomplete.value,
                                    note: document.getElementById('note').value,
                                };
        
                                fetch(`${localStorage.getItem('baseUrl')}/api/${roles.indexOf('ROLE_GESTSTOCK') === -1 ? 'point_of_sale_histories' : 'depot_histories'}`, {
                                    method: 'POST',
                                    headers: historyHeaders,
                                    body: JSON.stringify(history)
                                }).then((historyResp) => {
                                    if (historyResp.ok) {
                                        historyResp.text().then((body) => {
                                            const historyData = JSON.parse(body);
                                            const currentRow = stocks.find(row => row.id === data.id);
        
                                            if (typeof currentRow !== 'undefined') {
                                                Object.assign(currentRow, { quantity: newQty });
                                                stocksTableInstance.update({ rows: setStocks(stocks) });
                                            }
        
                                            document.getElementById('stock').value = '';
                                            spinner.setAttribute('hidden', 'hidden');
                                            destockingModal.hide();
                                            showAlert('Destockage effectué avec succès', 'success', false);
                                        });
                                    } else {
                                        Object.assign(depotDetail, { quantity: oldQty });
                                        fetch(stockUrl, {
                                            method: 'PUT',
                                            body: JSON.stringify(depotDetail),
                                            headers: stockHeaders
                                        }).then((nextResp) => {
                                            if (nextResp.ok) {
                                                nextResp.text().then((nextBody) => {
                                                    spinner.setAttribute('hidden', 'hidden');
                                                    destockingModal.hide();
                                                    showAlert('Echec de l\'opération', 'danger', false);
                                                });
                                            } else {
                                                spinner.setAttribute('hidden', 'hidden');
                                                destockingModal.hide();
                                                showAlert('Un erreur s\'est produite dans le système. Veuillez contacter l\'administrateur.', 'danger', false);
                                            }
                                        }).catch((erreur) => {
                                            spinner.setAttribute('hidden', 'hidden');
                                            destockingModal.hide();
                                            showAlert('Impossible de se connecter au serveur !', 'danger', false);
                                        });
                                    }
                                }).catch((error) => {
                                    spinner.setAttribute('hidden', 'hidden');
                                    destockingModal.hide();
                                    showAlert('Impossible de se connecter au serveur !', 'danger', false);
                                });
                            });
                        } else {
                            spinner.setAttribute('hidden', 'hidden');
                            destockingModal.hide();
                            showAlert('Un erreur s\'est produite dans le système.', 'danger', false);
                        }
                    }).catch((err) => {
                        spinner.setAttribute('hidden', 'hidden');
                        destockingModal.hide();
                        showAlert('Impossible de se connecter au serveur !', 'danger', false);
                    });
                });
            } else {
                showAlert('Une erreur s\'est produite dans le systeme', 'danger', false);
            }
        });

        
    }
});

