<?php

namespace App\Controller;

use App\AbstractController;
use App\Entity\Facture;
use App\Entity\Licence;
use App\Entity\Category;
use App\Entity\ControleTechnique;
use App\Entity\Service;
use App\Entity\DetailsFacture;
use App\Entity\Maison;
use App\Entity\ObservationControleTechnique;
use App\Entity\Vehicule;
use App\Tools\Exporter;
use App\Tools\VehicleUtils;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Security\Core\Authorization\AuthorizationCheckerInterface;
use Symfony\Component\String\Slugger\SluggerInterface;

class ControleTechniqueController extends AbstractController
{
    public const FIELDS = [
        ['name' => 'name', 'label' => 'Anomalie constatée'],
        ['name' => 'quantity', 'label' => 'Qté'],
        ['name' => 'observation', 'label' => 'Observation'],
    ];

    private $vehicleUtils;
    private $slugger;

    public function __construct(
        AuthorizationCheckerInterface $securityChecker,
        RequestStack $requestStack,
        private EntityManagerInterface $entityManager,
        VehicleUtils $vehicleUtils,
        SluggerInterface $slugger
    )
    {
        parent::__construct($securityChecker, $requestStack);
        $this->vehicleUtils = $vehicleUtils;
        $this->slugger = $slugger;
    }

    /**
     * @Route("/controletechnique/new/{id}", name="create_controletechnique")
     */
    public function create(Request $request, $id): Response
    {
        return $this->edit($request, $id);
    }

    /**
     * @Route("/controletechnique/{id}", name="update_controletechnique")
     */
    public function update(Request $request, $id): Response
    {
        return $this->edit($request, $id, true);
    }

    /**
     * @Route("/controlexpertise/{type}/{id}/export-pdf", name="export_pdf_controlexpertise")
     */
    public function exportPdf(Exporter $exporter, $type, $id)
    {
        $isControle = $type === 'control';
        $currentUser = $this->getUser();
        $vehicle = $this->entityManager->getRepository($isControle ? '\App\Entity\Licence' : '\App\Entity\Expertise')->find((int) $id)->getVehicule();

        $controls = $this->entityManager
            ->getRepository($isControle ? '\App\Entity\ControleTechnique' : '\App\Entity\DetailsExpertise')
            ->createQueryBuilder('c')
            ->select('l.id as licenseId', 's.name', 'c.quantity', 'c.observation', 'c.expert')
            ->where('l.id = :licenseId')
            ->innerJoin($isControle ? 'c.licence' : 'c.expertise', 'l')
            ->innerJoin('c.service', 's')
            ->innerJoin('s.category', 'cat')
            ->setParameter('licenseId', $id)
            ->getQuery()
            ->getResult()
        ;

        $company = $this->entityManager
            ->getRepository(Maison::class)
            ->createQueryBuilder('c')
            ->select('c.logo', 'c.nomSociete')
            ->getQuery()
            ->getSingleResult()
        ;

        $header = [
            'company' => $company,
            'vehicle' => [
                'genre' => $vehicle->getGenre(),
                'marque' => $vehicle->getMarque(),
                'typeOuModele' => $vehicle->getTypeOuModele(),
                'numeroImmatriculation' => $vehicle->getNumeroImmatriculation(),
                'kilometrage' => $vehicle->getKilometrage(),
            ],
            'client' => $vehicle->getClient()->getNomClient(),
        ];

        $expert = \count($controls) > 0 ? $controls[0]['expert'] : ($currentUser->getFullName() ?? $currentUser->getUsername());
        $fileName = sprintf('Rapport %s technique ', ($type === 'control' ? 'de contrôle' : 'd\'expertise')).$header['vehicle']['genre'].' '.$header['vehicle']['numeroImmatriculation'];

        $exporter->exportPdf(
            'export/controle-expertise.html.twig',
            $controls,
            [self::FIELDS[0], self::FIELDS[1], self::FIELDS[2]],
            $fileName,
            [
                'pageFooter' => '{DATE F j, Y}|'.$fileName.'|Page {PAGENO} of {nb}',
                'header' => $header,
                'expert' => $expert,
                'isControle' => $isControle
            ]
        );
    }

    private function edit(Request $request, $id, $edit = false): Response
    {
        $entityManager = $this->getDoctrine()->getManager();
        $licenceRepository = $entityManager->getRepository(Licence::class);
        $serviceRepository = $entityManager->getRepository(Service::class);
        $controleRepository = $entityManager->getRepository(ControleTechnique::class);
        $detailsFactureRepository = $entityManager->getRepository(DetailsFacture::class);
        $factureRepository = $entityManager->getRepository(Facture::class);
        $licenseId = \intval($id);
        $licence = $licenceRepository->find($licenseId);
        $newInvoice = false;

        if ($request->isMethod('POST')) {
            $total = 0;
            $facture = null;

            // find or create proforma invoice
            if ($edit) {
                // $facture = $factureRepository->findOneBy(['licence' => $licence, 'typeFacture' => 'proforma_ctrl']);
                $facture = $factureRepository
                    ->createQueryBuilder('i')
                    ->where('l.id = :licenseId AND i.typeFacture = :invoiceType')
                    ->innerJoin('i.licence', 'l')
                    ->setParameter('licenseId', $licenseId)
                    ->setParameter('invoiceType', 'proforma_ctrl')
                    ->getQuery()
                    ->getOneOrNullResult()
                ;
            }

            if (\is_null($facture)) {
                $newInvoice = true;
                $facture = new Facture();
            }

            $data = json_decode(file_get_contents('php://input'), true);
            $issues = $data['issues'];

            if (!$newInvoice) {
                $controls = $controleRepository->findBy(['licence' => $licence]);
                $servicesIds = [];

                if (count($controls) > 0) {
                    foreach ($controls as $control) {
                        $servicesIds[] = $control->getService()->getId();
                        $licence->removeControleTechnique($control);
                        $control->getService()->removeControleTechnique($control);
                        $control->getDoneBy()->removeControleTechnique($control);
                        $entityManager->remove($control);
                    }
                }

                $detailsFacturesIds = $detailsFactureRepository
                    ->createQueryBuilder('d')
                    ->select('d.id')
                    ->where('s.id IN(:servicesIds)')
                    ->andWhere('f.id = :factureId')
                    ->innerJoin('d.service', 's')
                    ->innerJoin('d.facture', 'f')
                    ->setParameter('servicesIds', $servicesIds)
                    ->setParameter('factureId', $facture->getId())
                    ->getQuery()
                    ->getResult()
                ;

                if (count($detailsFacturesIds) > 0) {
                    foreach ($detailsFacturesIds as $detailsFactureId) {
                        $detailsFacture = $detailsFactureRepository->find($detailsFactureId);
                        $detailsFacture->getService()->removeDetailsFacture($detailsFacture);
                        $detailsFacture->getFacture()->removeDetailsFacture($detailsFacture);
                        $entityManager->remove($detailsFacture);
                    }
                }
            }

            foreach ($issues as $issue) {
                $serviceId = $issue['id'];

                if ($serviceId > 0) {
                    $service = $serviceRepository->find($serviceId);
                    $quantity = floatval($issue['quantity']);
                    $price = $quantity * $service->getUnitPrice();
                    $total += $price;
                    $obsName = trim($issue['observation']);
                    $observation = $entityManager->getRepository(ObservationControleTechnique::class)->findOneBy(['name' => $obsName]);

                    if (is_null($observation)) {
                        $observation = new ObservationControleTechnique();
                        $slug = $this->slugger->slug(strtolower($obsName), '-', 'en');
                        $observation->setSlug($slug);
                        $observation->setName($obsName);
                        $entityManager->persist($observation);
                    }

                    $ctrl = new ControleTechnique();
                    $ctrl->setQuantity($quantity);
                    $ctrl->setObservation($observation->getName());
                    $ctrl->setExpert(trim($data['expert']) !== '' ? $data['expert'] : ($this->getUser()->getFullName() ?? $this->getUser()->getUserIdentifier()));
                    $service->addControleTechnique($ctrl);
                    $licence->addControleTechnique($ctrl);
                    $this->getUser()->addControleTechnique($ctrl);

                    $detailsFacture = new DetailsFacture();
                    $detailsFacture->setQty($quantity);
                    $detailsFacture->setPrice($price);
                    // $pos->addPointOfSaleHistory($detailsFacture);
                    $service->addDetailsFacture($detailsFacture);
                    $facture->addDetailsFacture($detailsFacture);

                    $entityManager->persist($ctrl);
                    $entityManager->persist($detailsFacture);
                }
            }

            $facture->setTotal($total);

            if ($newInvoice) {
                $facture->setTypeFacture('proforma_ctrl');
                $licence->addFacture($facture);
                $licence->getVehicule()->getClient()->addFacture($facture);
                $this->getUser()->addFacture($facture);
                $entityManager->persist($facture);
            }

            $entityManager->flush();

            $controls = $controleRepository
                ->createQueryBuilder('c')
                ->select('c.id as controlId', 's.id', 's.name', 'c.quantity', 'c.observation', 'c.expert', 'cat.id as categoryId')
                ->where('l.id = :licenseId')
                ->innerJoin('c.licence', 'l')
                ->innerJoin('c.service', 's')
                ->innerJoin('s.category', 'cat')
                ->setParameter('licenseId', $id)
                ->getQuery()
                ->getResult()
            ;

            $observations = $entityManager
                ->createQuery("SELECT o.id, o.slug, o.name FROM App\Entity\ObservationControleTechnique o")
                ->getResult()
            ;

            return new JsonResponse(['facture' => $facture, 'controls' => $controls, 'observations' => $observations]);
        }

        $vehicule = $licence->getVehicule();
        $controls = [];
        $category = null;

        if (!$newInvoice) {
            $controls = $controleRepository
                ->createQueryBuilder('c')
                ->select('c.id as controlId', 's.id', 's.name', 'c.quantity', 'c.observation', 'c.expert', 'cat.id as categoryId')
                ->where('l.id = :licenseId')
                ->innerJoin('c.licence', 'l')
                ->innerJoin('c.service', 's')
                ->innerJoin('s.category', 'cat')
                ->setParameter('licenseId', $id)
                ->getQuery()
                ->getResult()
            ;

            if (count($controls) > 0) {
                $category = $entityManager
                    ->getRepository(Category::class)
                    ->createQueryBuilder('c')
                    ->select('c.name')
                    ->where('c.id = :categoryId')
                    ->setParameter('categoryId', $controls[0]['categoryId'])
                    ->getQuery()
                    ->getOneOrNullResult()
                ;
            }
        }

        $maison = $entityManager
            ->createQuery("SELECT m.logo, m.nomSociete, m.registreCommerce, m.idNationnal, m.numImpot, m.autorisationFct, m.adressePhysique, m.telephone, m.mail, m.site FROM App\Entity\Maison m")
            ->getOneOrNullResult()
        ;

        return $this->render('controletechnique/edit.html.twig', [
            'title' => 'Controle Technique',
            'breadcrumb' => [
                ['name' => 'P.V', 'path' => '/licences'],
                ['name' => isset($id) ? $licenceRepository->find($id)->getNumeroPv() : 'Controle Technique']
            ],
            'licenseId' => $id,
            'licence' => $licence,
            'vehicule' => $this->vehicleUtils->toArray($vehicule),
            'client' => $vehicule->getClient()->getNomClient(),
            'controls' => $controls,
            'category' => (is_null($category) ? null : $category['name']),
            'maison' => $maison
        ]);
    }
}
