<?php

namespace App\Controller;

use App\Entity\Client;
use App\AbstractController;
use App\Entity\Affectation;
use App\Entity\Caution;
use App\Entity\Currency;
use App\Entity\Facture;
use App\Entity\Dettes;
use App\Entity\Facturedateretrait;
use App\Entity\Maison;
use App\Entity\PayementDette;
use App\Entity\PointOfSaleDetails;
use App\Entity\PointOfSaleHistory;
use App\Entity\Service;
use App\Entity\UserGuichet;
use App\Http\ApiResponse;
use App\Tools\Exporter;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;

class FacturesClientController extends AbstractController
{
    /**
     * @Route("/facturesclient", name="list_factures")
     */
    public function list(Request $request)
    {
        $em = $this->getDoctrine()->getManager();
        $cl = $em->getRepository(Client::class);
        $fact = $em->getRepository(Facture::class);
        $latestClients = $cl->findBy([], ['nomClient' => 'ASC']);

        return $this->render('facturesclient/list.html.twig', [
            'title' => 'FACTURES CLIENTS',
            'breadcrumb' => [
                [
                    'name' => 'Factures client',
                    'path' => '/facturesclient'
                ]
            ],
            'latestClients' => $latestClients
        ]);
    }

    /**
     * @Route("/facturesclients/{id}",name="single_factureclient")
     */
    public function getfacturesclient(Request $request, int $id): Response
    {
        $manager = $this->getDoctrine()->getManager();
        $cr = $manager->getRepository(Client::class);
        $fr = $manager->getRepository(Facture::class);
        $client = $cr->find($id);

        $factClients = $fr->createQueryBuilder('f')
            ->where('c.id = :id')
            ->innerJoin('f.client', 'c')
            ->setParameter('id', $id)
            ->getQuery()
            ->getResult()
        ;

        $currency = $manager->getRepository(Currency::class)->findOneBy(['main' => true]);

        $data = ['facturesclient' => $factClients, 'client' => $client, 'currency' => $currency->getSymbol()];

        if ($request->isXMLHttpRequest()) {
            return new JsonResponse($factClients);
        }

        return $this->render('facturesclient/singledetail.html.twig', [
            'title' => 'FACTURES CLIENT',
            'breadcrumb' => [
                [
                    'name' => 'Factures client',
                    'path' => '/facturesclient'
                ]
            ],
            'datasent' => $data
        ]);
    }

    /**
     * @Route("/facturesclient/new",name="new_factureclient")
     */
    public function new(Request $request): Response
    {
        $em = $this->getDoctrine()->getManager();
        $cl = $em->getRepository(Client::class);
        $clients = $cl->findBy([], ['nomClient' => 'ASC']);
        $invoices = [];
        $selectedClient = null;
        $selectedFrom = null;
        $selectedTo = null;

        if ($request->isMethod('POST')) {
            $invoiceRepository = $em->getRepository(Facture::class);
            $from = $request->request->get('du');
            $to = $request->request->get('au');
            $clientId = \intval($request->request->get('client'));

            $qb = $invoiceRepository->createQueryBuilder('i')->where('c.id = :clientId');

            if (\trim($from) !== '') {
                $selectedFrom = $from; // (new \DateTime($from))->format('Y-m-d');
                $qb = $qb->andWhere('i.dateFact >= :from')->setParameter('from', $selectedFrom);
            }

            if (\trim($to) !== '') {
                $selectedTo = $to; //(new \DateTime($to))->format('Y-m-d');
                $qb = $qb->andWhere('i.dateFact <= :to')->setParameter('to', $selectedTo);
            }

            $invoices = $qb->innerJoin('i.client', 'c')
                ->setParameter('clientId', $clientId)
                ->getQuery()
                ->getResult()
            ;

            $selectedClient = \array_values(\array_filter($clients, fn ($c) => $c->getId() == $clientId))[0];
        }

        return $this->render('facturesclient/new.html.twig', [
            'title' => 'FACTURES CLIENTS',
            'breadcrumb' => [
                [
                    'name' => 'Factures client',
                    'path' => '/facturesclient/new'
                ]
            ],
            'clients' => $clients,
            'invoices' => $invoices,
            'selectedClient' => $selectedClient,
            'selectedFrom' => $selectedFrom,
            'selectedTo' => $selectedTo
        ]);
    }

    /**
     * @Route("/factures/updatetotalelements/{id}", name="updatetotalelements")
     */
    public function updatetotalelements($id): JsonResponse
    {
        $invoice = $this->getDoctrine()->getManager()->getRepository(Facture::class)->find(intval($id));

        if (null !== $invoice) {
            $client = $invoice->getClient();
            $license =  $invoice->getLicence();
            $clientId = null !== $client ? $client->getId() : 0;
            $licenseId = null !== $license ? $license->getId() : 0;
        } else {
            $clientId = 0;
            $licenseId = 0;
        }
        
        // return new JsonResponse(['total' => $somme, 'tax' => $taxe, 'net' => $net, 'clientid' => $clientid, 'licenceid' => $licenceid]);
        return new JsonResponse(['clientId' => $clientId, 'licenseId' => $licenseId]);
    }

    /**
     * @Route("/factures/{id}", name="detailfacturesclient_client")
     */
    public function detail($id): Response
    {
        $manager = $this->getDoctrine()->getManager();
        $fr = $manager->getRepository(Facture::class);
        $phr = $manager->getRepository(PointOfSaleHistory::class);

        $qb = $phr->createQueryBuilder('p');
        $histories = $qb
            ->where('f.id = :id')
            ->andWhere($qb->expr()->like('p.motifDestockage', $qb->expr()->literal('vente%')))
            ->innerJoin('p.invoice', 'f')
            ->setParameter('id', $id)
            ->getQuery()
            ->getResult();

        return $this->render('facturesclient/details.html.twig', [
            'title' => 'Détails de la facture N° '.$id,
            'breadcrumb' => [
                [
                    'name' => 'Factures client',
                    'path' => '/facturesclient/new'
                ],
                [
                    'name' => 'Détails de la facture N° '.$id
                ]
            ],
            'histories' => $histories,
            'facture' => $fr->find(intval($id))
        ]);
    }

    /**
     * @Route("/factures/vente/guichetier", name="FacturesGechetierList")
     */
    public function guichetierFactures(Request $request): JsonResponse
    {
        $invoiceDate = $request->get('date');
        $searchQuery = $request->get('search');
        $em = $this->getDoctrine()->getManager();
        $invoiceRepository = $em->getRepository(Facture::class);
        $queryBuilder = $invoiceRepository->createQueryBuilder('f');
        $queryBuilder = $queryBuilder
            ->select('f.id', 'f.typeFacture', 'f.dateFact', 'f.total', 'f.vatAmount', 'f.workforce', 'f.discount', 'c.nomClient')
            ->where($queryBuilder->expr()->neq('f.typeFacture', ':typeFacture'))
        ;

        if ($invoiceDate) {
            $queryBuilder = $queryBuilder->andWhere($queryBuilder->expr()->eq("DATE_FORMAT(f.dateFact, '%Y-%m-%d')", ':invoiceDate'));
        }

        if ($searchQuery) {
            $invoiceNumber = intval($searchQuery);

            if (is_int($invoiceNumber) && $invoiceNumber > 0) {
                $queryBuilder = $queryBuilder->andWhere($queryBuilder->expr()->orX(
                    $queryBuilder->expr()->eq('f.id', ':searchQuery'),
                    $queryBuilder->expr()->eq('f.id', ':invoiceNumber')
                ));
            } else {
                $queryBuilder = $queryBuilder->andWhere($queryBuilder->expr()->orX(
                    $queryBuilder->expr()->like('c.nomClient', ':searchQuery'), $queryBuilder->expr()->orX(
                        $queryBuilder->expr()->like('f.typeFacture', ':searchQuery'), $queryBuilder->expr()->like("DATE_FORMAT(f.dateFact, '%Y-%m-%d')", ':searchQuery')
                    )
                ));
            }
        }

        if ($this->getUser()->getType() !== 'super_guichetier') {
            $queryBuilder = $queryBuilder
                ->andWhere($queryBuilder->expr()->orX($queryBuilder->expr()->in('f.typeFacture', ['proforma', 'proforma_ctrl']), 'u = :user'))
                ->leftJoin('f.client', 'c')
                ->innerJoin('f.editedBy', 'u')
                ->setParameter('user', $this->getUser())
            ;
        } else {
            $affectationRepository = $em->getRepository(Affectation::class);
            $pos = $affectationRepository->findOneBy(['user' => $this->getUser()])->getPos();
            $usersIds = $affectationRepository
                ->createQueryBuilder('a')
                ->select('u.id')
                ->where('p = :pos')
                ->innerJoin('a.pos', 'p')
                ->innerJoin('a.user', 'u')
                ->setParameter('pos', $pos)
                ->getQuery()
                ->getResult()
            ;
            $queryBuilder = $queryBuilder
                ->andWhere($queryBuilder->expr()->orX($queryBuilder->expr()->in('f.typeFacture', ['proforma', 'proforma_ctrl']), $queryBuilder->expr()->in('u.id', ':usersIds')))
                ->leftJoin('f.client', 'c')
                ->innerJoin('f.editedBy', 'u')
                ->setParameter('usersIds', $usersIds)
            ;
        }

        if ($invoiceDate) {
            $queryBuilder = $queryBuilder->setParameter('invoiceDate', $invoiceDate);
        }

        if ($searchQuery) {
            if (isset($invoiceNumber) && is_int($invoiceNumber) && $invoiceNumber > 0) {
                $queryBuilder = $queryBuilder->setParameter('invoiceNumber', $invoiceNumber);
                $queryBuilder = $queryBuilder->setParameter('searchQuery', $searchQuery);
            } else {
                $queryBuilder = $queryBuilder->setParameter('searchQuery', '%'.$searchQuery.'%');
            }
        }

        $factures = $queryBuilder
            ->setParameter('typeFacture', 'cancelled')
            ->orderBy('f.dateFact', 'DESC')
            ->getQuery()
            ->getResult()
        ;

        return new JsonResponse($factures);
    }

    /**
     * @Route("/factures/{id}/populate/cartboard", name="populatecartboard")
     */
    public function facturedetails($id): JsonResponse
    {
        $entityManager = $this->getDoctrine()->getManager();
        $invoice = $entityManager->getRepository(Facture::class)->find(intval($id));

        if (in_array($invoice->getTypeFacture(), ['cash', 'credit'], true)) {
            $entity = 'App\Entity\PointOfSaleHistory';
            $quantityProperty = 'p.quantity as qty';
            $invoiceRelation = 'p.invoice';
        } else {
            $entity = 'App\Entity\DetailsFacture';
            $quantityProperty = 'p.qty';
            $invoiceRelation = 'p.facture';
        }

        $queryString = " s.id as serviceId, s.bonusApplicable, c.name as category, s.unitPrice, s.deflatedPrice, s.wholesalePrice, s.hasVat, p.price as selectedPrice, " . $quantityProperty . ", s.type, s.name, s.photo";
        $queryStringMiddle = ", pos.id as depotId, pos.name as depotName FROM " . $entity . " p INNER JOIN p.service AS s INNER JOIN s.category AS c LEFT JOIN p.pos AS pos";
        // $queryStringMiddle = " FROM " . $entity . " p INNER JOIN p.service AS s INNER JOIN s.category AS c";
        $queryStringTail = " WHERE " . $invoiceRelation . " = :invoice";

        $articles = $entityManager
            ->createQuery("SELECT pd.id," . $queryString . ", pd.quantity, pd.codeBars" . $queryStringMiddle . " INNER JOIN s.pointOfSaleDetails AS pd" . $queryStringTail . " AND s.type = 2")
            ->setParameter('invoice', $invoice)
            ->getResult()
        ;

        $services = $entityManager
            ->createQuery("SELECT" . $queryString . $queryStringMiddle . $queryStringTail . " AND s.type = 1")
            ->setParameter('invoice', $invoice)
            ->getResult()
        ;

        return new JsonResponse(['message' => '', 'data' => ['services' => array_merge($articles, $services), 'typeFacture' => $invoice->getTypeFacture(), 'paymentMode' => $invoice->getPaymentMode(), 'workforce' => $invoice->getWorkforce(), 'workforcePercent' => $invoice->getWorkforcePercent()], 'errors' => []]);
    }

    /**
     * @Route("/annulerfacture", name="cancel_invoice")
     */
    public function annulerFacture(Request $request): Response
    {
        $entityManager = $this->getDoctrine()->getManager();
        $affectationRepository = $entityManager->getRepository(Affectation::class);
        $invoiceRepository = $entityManager->getRepository(Facture::class);
        $posHistoryRepository = $entityManager->getRepository(PointOfSaleHistory::class);
        $posDetailRepository = $entityManager->getRepository(PointOfSaleDetails::class);
        $debtRepository = $entityManager->getRepository(Dettes::class);
        $debtPaymentsRepository = $entityManager->getRepository(PayementDette::class);

        if ($request->isMethod('POST')) {
            $invoiceId = json_decode(file_get_contents('php://input'), true)['invoiceId'];
            $invoice = $invoiceRepository->find(intval($invoiceId));
            $posHistories = $posHistoryRepository->findBy(['invoice' => $invoice]);
            
            if (count($posHistories) > 0) {
                $services = [];
                $pos = $affectationRepository->findOneBy(['user' => $invoice->getEditedBy()])->getPos();

                foreach ($posHistories as $posHistory) {
                    $service = $posHistory->getService();
                    if ($service->getType() === 2) {
                        $serviceName = $service->getName();
                        if (!array_key_exists($serviceName, $services)) {
                            $services[$serviceName] = ['service' => $serviceName, 'quantity' => abs($posHistory->getQuantity()), 'price' => $service->getUnitPrice()];
                        } else {
                            $services[$serviceName]['quantity'] += abs($posHistory->getQuantity());
                        }
                    }
                }
                
                foreach ($services as $service) {
                    $posDetail = $posDetailRepository->findOneBy(['pointOfSale' => $pos, 'service' => $entityManager->getRepository(Service::class)->findOneBy(['name' => $service['service']])]);
                    $posDetail->setQuantity($posDetail->getQuantity() + $service['quantity']);
                    $posDetail->setValue($posDetail->getValue() + $service['quantity'] * $service['price']);
                }

                if ($invoice->getTypeFacture() === 'credit') {
                    $debt = $debtRepository->findOneBy(['facture' => $invoice]);
                    $debtPayments = $debtPaymentsRepository->findBy(['dette' => $debt]);

                    foreach ($debtPayments as $debtPayment) {
                        $debtPayment->getDoneby()->removePayementDette($debtPayment);
                        $debtPayment->getDette()->removePayementDette($debtPayment);
                        $entityManager->remove($debtPayment);
                    }

                    $debt->getMadeby()->removeDette($debt);
                }
            }

            $invoice->setTypeFacture('cancelled');
            $entityManager->flush();
            $invoiceId = $invoice->getId();

            return new ApiResponse(sprintf("Facture N°%d annulée avec succès", $invoiceId), ['invoice' => $invoiceId]);
        }

        $queryBuilder = $invoiceRepository->createQueryBuilder('i');
        $invoices = $queryBuilder
            ->select('i.id', 'i.dateFact', 'i.typeFacture', 'i.total', 'c.nomClient')
            ->where($queryBuilder->expr()->neq('i.typeFacture', '?1'))
            ->leftJoin('i.client', 'c')
            ->setParameter('1', 'cancelled')
            ->getQuery()
            ->getResult()
        ;

        $currency = $entityManager->getRepository(Currency::class)->findOneBy(['main' => true]);

        return $this->render('factures/cancelation.html.twig', [
            'title' => 'Annulation de factures',
            'breadcrumb' => [
                [
                    'name' => 'Annulation de factures'
                ]
            ],
            'invoices' => $invoices,
            'currency' => $currency->getSymbol(),
        ]);
    }

    /**
     * @Route("/facturesannulees", name="cancelled_invoices")
     */
    public function facturesAnnulees(Request $request): Response
    {
        $factures = $this
            ->getDoctrine()
            ->getManager()
            ->getRepository(Facture::class)
            ->createQueryBuilder('f')
            ->select('f.id', 'f.typeFacture', 'f.dateFact', 'f.total', 'f.discount', 'c.nomClient')
            ->where('f.typeFacture = :type')
            ->leftJoin('f.client', 'c')
            ->setParameter('type', 'cancelled')
            ->getQuery()
            ->getResult()
        ;

        return $this->render('factures/facturesannulees.html.twig', [
            'title' => 'Factures annulées',
            'breadcrumb' => [
                ['name' => 'Factures annulées']
            ],
            'factures' => $factures
        ]);
    }

    /**
     * @Route("/facturedateretrait", name="facturedateretrait")
     */
    public function facturedateretrait(): JsonResponse
    {
        $entityManager = $this->getDoctrine()->getManager();
        $invoiceRepository = $entityManager->getRepository(Facture::class);
        $dateRetraitRepository = $entityManager->getRepository(Facturedateretrait::class);

        $data = json_decode(file_get_contents('php://input'), true); 
        $factureId = $data['factureId'];
        $invoice = $invoiceRepository->find(intval($factureId));
        $entry = $dateRetraitRepository->findOneBy(['invoice' => $invoice]);

        if ($entry == null) {
            $date = trim($data['date']); // $data['date'] est sous la forme '2021-10-15T19:30'
            $entry = new Facturedateretrait();
            $entry->setDateRetrait(new \DateTime($date));
            $invoiceRepository->find(intval($factureId))->addFacturedateretrait($entry);

            $entityManager->persist($entry);
            $entityManager->flush();
        }

        $client = $invoice->getClient();
        $license = $invoice->getLicence();
        $vehicle = null !== $license ? $license->getVehicule() : null;
        $totalBonus = $entityManager
            ->createQuery("SELECT SUM(b.amount - b.amountUsed) FROM App\Entity\Bonus b WHERE b.client = :client")
            ->setParameter('client', $client)
            ->getSingleScalarResult()
        ;

        return new JsonResponse([
            'id' => $entry->getId(),
            'dateRetrait' => $entry->getDateRetrait(),
            'invoice' => [
                'id' => $invoice->getId(),
                'total' => $invoice->getTotal(),
                'discount' => $invoice->getDiscount(),
                'amountPaid' => $invoice->getAmountPaid(),
                'workforce' => $invoice->getWorkforce(),
                'workforcePercent' => $invoice->getWorkforcePercent(),
                'type' => $invoice->getTypeFacture(),
                'paymentMode' => $invoice->getPaymentMode(),
                'licenceId' => (null !== $license ? $license->getId() : null),
                'date' => $invoice->getDateFact()
            ],
            'marque' => (null !== $vehicle ? $vehicle->getMarque() : null),
            'typeOuModele' => (null !== $vehicle ? $vehicle->getTypeOuModele() : null),
            'numeroImmatriculation' => (null !== $vehicle ? $vehicle->getNumeroImmatriculation() : null),
            'kilometrage' => (null !== $vehicle ? $vehicle->getKilometrage() : null),
            'client' => ['id' => (null !== $client ? $client->getId() : null), 'nomClient' => (null !== $client ? $client->getNomClient() : null), 'telephone' => (null !== $client ? $client->getTelephone() : null), 'adressePhys' => (null !== $client ? $client->getAdressePhys() : null), 'totalBonus' => (is_string($totalBonus) ? floatval($totalBonus) : $totalBonus)]
        ]);
    }

    /**
     * @Route("/invoices/dailyreport", name="invoices_daily_report")
     */
    public function dailyReport(): JsonResponse
    {
        $entityManager = $this->getDoctrine()->getManager();
        $invoiceRepository = $entityManager->getRepository(Facture::class);
        $debtPaymentsRepository = $entityManager->getRepository(PayementDette::class);
        $posHistoryRepository = $entityManager->getRepository(PointOfSaleHistory::class);
        $cautionRepository = $entityManager->getRepository(Caution::class);
        $now = new \DateTime();
        $today = $now->format("Y-m-d");
        $unixEpoch = new \DateTime('1970-01-01 00:00:00');
        $todayMidnight = new \DateTime($today." 00:00:00");

        $guichet = $entityManager->getRepository(UserGuichet::class)->findOneBy(['user' => $this->getUser()])->getGuichet();
        $pos = $guichet->getPos();

        $qBuilder = $posHistoryRepository
            ->createQueryBuilder('ph')
            ->select('ph.quantity', 'ph.motifDestockage', 's.name as article', 's.hasVat')
        ;
        $qBuild = $posHistoryRepository
            ->createQueryBuilder('ph')
            ->select('ph.quantity', 'ph.motifDestockage', 's.name as article', 's.hasVat')
        ;
        $tsQb = $posHistoryRepository
            ->createQueryBuilder('ph')
            ->select('ph.quantity', 'ph.motifDestockage', 's.name as article', 's.hasVat')
        ;
        $osQb = $posHistoryRepository
            ->createQueryBuilder('ph')
            ->select('ph.quantity', 'ph.motifDestockage', 's.name as article', 's.hasVat')
        ;
        $cQb = $cautionRepository
            ->createQueryBuilder('c')
            ->select('c.id', 'c.amount', 'c.amountUsed', 'cl.nomClient')
        ;
        $cautions = $cQb
            ->where($cQb->expr()->andX($cQb->expr()->eq('c.user', ':user'), $cQb->expr()->eq("DATE_FORMAT(c.createdAt, '%Y-%m-%d')", ':today')))
            ->innerJoin('c.client', 'cl')
            ->setParameter('user', $this->getUser())
            ->setParameter('today', $today)
            ->orderBy('c.createdAt', 'DESC')
            ->getQuery()
            ->getResult()
        ;
        $procurements = $qBuilder
            ->where('p = :pos')
            ->andWhere($qBuilder->expr()->isNull('ph.motifDestockage'))
            // ->andWhere($qBuilder->expr()->notIn('ph.motifDestockage', ['vente_cash', 'vente_credit']))
            // ->groupBy('s.name')
            ->innerJoin('ph.pos', 'p')
            ->innerJoin('ph.service', 's')
            ->setParameter('pos', $pos)
            ->getQuery()
            ->getResult()
        ;
        $posDetails = $entityManager
            ->getRepository(PointOfSaleDetails::class)
            ->createQueryBuilder('pd')
            ->select('pd.quantity', 's.name as article', 's.hasVat')
            ->where('p = :pos')
            ->innerJoin('pd.service', 's')
            ->innerJoin('pd.pointOfSale', 'p')
            ->setParameter('pos', $pos)
            ->getQuery()
            ->getResult()
        ;
        $outOfStock = $osQb
            ->where($osQb->expr()->andX($osQb->expr()->isNotNull('ph.motifDestockage'), $osQb->expr()->notIn('ph.motifDestockage', ['vente_cash', 'vente_credit'])))
            ->andWhere('s.type = 2')
            ->andWhere('p = :pos')
            ->innerJoin('ph.service', 's')
            ->innerJoin('ph.pos', 'p')
            ->setParameter('pos', $pos)
            ->getQuery()
            ->getResult()
        ;
        $sales = $qBuild
            ->andWhere($qBuild->expr()->in('ph.motifDestockage', ['vente_cash', 'vente_credit']))
            ->andWhere('u = :user')
            ->andWhere('s.type = 2')
            ->andWhere($qBuild->expr()->neq('f.typeFacture', ':cancelled'))
            ->innerJoin('ph.service', 's')
            ->innerJoin('ph.invoice', 'f')
            ->innerJoin('f.editedBy', 'u')
            ->setParameter('user', $this->getUser())
            ->setParameter('cancelled', 'cancelled')
            ->getQuery()
            ->getResult()
        ;
        $todaySales = $tsQb
            ->andWhere($tsQb->expr()->in('ph.motifDestockage', ['vente_cash', 'vente_credit']))
            ->andWhere($tsQb->expr()->eq('u.id', ':userId'))
            ->andWhere('s.type = 2')
            ->andWhere("DATE_FORMAT(f.dateFact, '%Y-%m-%d') = :today")
            ->andWhere($tsQb->expr()->neq('f.typeFacture', ':cancelled'))
            ->innerJoin('ph.service', 's')
            ->innerJoin('ph.invoice', 'f')
            ->innerJoin('f.editedBy', 'u')
            ->setParameter('userId', $this->getUser()->getId())
            ->setParameter('today', $today)
            ->setParameter('cancelled', 'cancelled')
            ->getQuery()
            ->getResult()
        ;

        $stock = ['procurements' => $procurements, 'posDetails' => $posDetails, 'outOfStock' => $outOfStock, 'sales' => $sales, 'todaySales' => $todaySales];

        $queryBuilder = $invoiceRepository
            ->createQueryBuilder('f')
            ->select('f.id', 'f.typeFacture', 'f.dateFact', 'f.total', 'f.discount', 'f.amountPaid', 'f.vatAmount', 'f.paymentMode', 'c.nomClient', 'c.telephone')
            ->where("DATE_FORMAT(f.dateFact, '%Y-%m-%d') = :today")
        ;

        $queryBuilder = $queryBuilder->andWhere($queryBuilder->expr()->notIn('f.typeFacture', ['proforma', 'proforma_ctrl', 'hold']));

        $qb = $debtPaymentsRepository
            ->createQueryBuilder('p')
            ->select('p.id', 'f.id as invoiceId', 'd.dateOctroi', 'd.statut', 'd.amount', 'd.sold', 'd.echeance', 'p.datePaye', 'p.montantPaye', 'c.nomClient')
            ->where("DATE_FORMAT(p.datePaye, '%Y-%m-%d') = :today")
            ->andWhere('f.dateFact BETWEEN :from AND :to')
        ;

        // if ($this->getUser()->getType() !== 'super_guichetier') {
            $queryBuilder = $queryBuilder
                ->andWhere($queryBuilder->expr()->eq('u.id', ':userId'))
                ->leftJoin('f.client', 'c')
                ->innerJoin('f.editedBy', 'u')
                ->setParameter('userId', $this->getUser()->getId())
            ;

            $qb = $qb
                ->andWhere($qb->expr()->eq('u.id', ':userId'))
                ->setParameter('userId', $this->getUser()->getId())
                ->innerJoin('p.doneby', 'u')
            ;
        // } else {
        //     $queryBuilder = $queryBuilder->innerJoin('f.client', 'c');
        // }

        $factures = $queryBuilder
            ->setParameter('today', $today)
            ->orderBy('f.dateFact', 'DESC')
            ->getQuery()
            ->getResult()
        ;

        $debtPayments = $qb
            ->innerJoin('p.dette', 'd')
            ->innerJoin('d.facture', 'f')
            ->innerJoin('d.client', 'c')
            ->setParameter('today', $today)
            ->setParameter('from', $unixEpoch)
            ->setParameter('to', $todayMidnight)
            ->orderBy('p.datePaye', 'DESC')
            ->getQuery()
            ->getResult()
        ;

        return new ApiResponse('', ['invoices' => $factures, 'debtPayments' => $debtPayments, 'cautions' => $cautions, 'stock' => $stock]);
    }

    /**
     * @Route("/invoices/{id}/export-pdf/{template_name}", name="export_pdf_invoice")
     */
    public function exportPdfInvoice(Request $request, EntityManagerInterface $entityManager, Exporter $exporter, ObjectNormalizer $objectNormalizer, $id, $template_name)
    {
        $invoiceRepository = $entityManager->getRepository(Facture::class);
        $currencyRepository = $entityManager->getRepository(Currency::class);
        $invoice = $invoiceRepository->find($id);
        $client = $invoice->getClient();
        $printType = $request->query->get('printType');
        $invoiceType = $invoice->getTypeFacture();
        $proformaInvoice = \str_contains($invoiceType, 'proforma');
        $fileName = \sprintf('%s %s %s', $printType, $invoice->getId(), $client?->getNomClient());
        $invoiceItems = $proformaInvoice ? $invoice->getDetailsFactures() : $invoice->getPointOfSaleHistories();
        $invoiceItemsCount = \count($invoiceItems);
        // dd($invoice, $invoice->getLicence());

        $company = $entityManager
            ->createQuery("SELECT m.logo, m.nomSociete, m.registreCommerce, m.idNationnal, m.numImpot, m.autorisationFct, m.adressePhysique, m.telephone, m.mail, m.site, m.vatRate FROM App\Entity\Maison m")
            ->getOneOrNullResult()
        ;

        $clientId = \is_null($client) ? null : $client->getId();
        $clientCode = \is_null($client) ? '' : 'C'.\substr(($clientId + 100000), 1);

        $header = [
            'company' => $company,
            'vehicle' => (\is_null($invoice->getLicence()) ? null : $invoice->getLicence()->getVehicule()),
            'client' => $client,
            'clientCode' => $clientCode,
            'invoiceDate' => $invoice->getDateFact()
        ];

        // $options['pageFooter'] = '{DATE F j, Y}|'.$fileName.'|Page {PAGENO} of {nb}';

        $total = $invoice->getTotal();
        $workforce = $invoice->getWorkforce();
        $discount = $invoice->getDiscount();
        $others = $workforce - (\is_null($client) ? 0 : $discount);
        $invoiceTtc = $others;
        $items = [];
        $productsWithBonus = [];
        $vatRate = $company['vatRate'];
        $companyHasVat = \is_float($vatRate) && $vatRate > 0;

        $currentUser = $this->getUser();
        $assignment = $entityManager->getRepository(Affectation::class)->findOneBy(['user' => $currentUser]);
        $branch = $assignment->getPointOfSale();
        $bonusThreshold = $branch->getNbSalesBonus();
        $bonusPercentage = $branch->getBonusPercentage();

        foreach ($invoiceItems as $item) {
            $product = $item->getService();
            $quantity = \abs($proformaInvoice ? $item->getQty() : $item->getQuantity());
            $price = $item->getPrice();
            // $totalPrice = $quantity * $price;
            $vatFactor = $this->getVatFactor($item->getService()->getHasVat(), $companyHasVat, $vatRate);
            $localPrice = $price * $vatFactor;
            $localTotalPrice = $quantity * $localPrice;
            $invoiceTtc += $localTotalPrice;
            $invoiceItem = ['quantity' => $quantity, 'price' => $localPrice, 'total' => $localTotalPrice, 'product' => $product];
            $items[] = $invoiceItem;

            if ($product->isBonusApplicable()) {
                $productsWithBonus[] = $invoiceItem;
            }
        }

        $invoiceNet = $total + $others;

        $options = [
            'header' => $header,
            // 'roll58' => [58, 1 * 6 + 18],
            // 'roll80' => [80, 1 * 6 + 18],
            'companyHasVat' => $companyHasVat,
            'currency' => $currencyRepository->findOneBy(['main' => true]),
            'itemsContainer' => $request->query->get('itemsContainer'),
            'invoiceItems' => $items,
            'invoiceNet' => $invoiceNet,
            'vatRate' => $vatRate,
            'invoiceTotal' => ($companyHasVat ? $invoiceTtc : $invoiceNet)
        ];

        if (\count($productsWithBonus) > 0) {
            if ($bonusThreshold > 1) {
                $options['note'] = '<div class="mt-2"><em>N.B.: Bonus'.(\is_float($bonusPercentage) && $bonusPercentage > 0 ? ' de '.\round($bonusPercentage * 100, 2).' %' : '').' pour '.$bonusThreshold.' produits de même nature</em></div>';
            } else if ($bonusThreshold == 1) {
                $options['note'] = '<div class="mt-2"><em>N.B.: Bonus de '.\round($bonusPercentage * 100, 2).' % pour tout achat</em></div>';
            }
        }

        $staticContentHeight = 58;
        $lineHeight = 3;

        if (!\is_null($client)) {
            $staticContentHeight += 5;
        }

        if (!\is_null($workforce)) {
            $staticContentHeight += 2;
        }

        if ($companyHasVat) {
            $staticContentHeight += 4;
        }

        if (!\is_null($discount)) {
            $staticContentHeight += 2;
        }

        switch ($template_name) {
            case 'roll58':
                $options['config']['format'] = [58, $invoiceItemsCount * $lineHeight + $staticContentHeight];
                $options['config']['default_font_size'] = 8;
                $options['config']['default_font'] = 'monospace';
                break;
            case 'roll80':
                $options['config']['format'] = [80, $invoiceItemsCount * $lineHeight + $staticContentHeight];
                $options['config']['default_font_size'] = 8;
                $options['config']['default_font'] = 'monospace';
                break;
            default:
                $options['config']['format'] = $template_name;
                break;
        }

        // $invoice = $objectNormalizer->normalize($invoice, 'json', ['groups' => ['invoice:read']]);
        // $invoiceNet = data.total + data.workforce - (data.client.id != null ? data.discount : 0)
        // dd($options);
        $exporter->exportPdf(
            'export/invoice/'.$template_name.'.html.twig',
            $invoice,
            [],
            $fileName,
            $options
        );
    }

    private function getVatFactor(bool $hasVat, bool $companyHasVat, mixed $vatRate): float
    {
        return 1 + ($hasVat && $companyHasVat ? $vatRate : 0);
    }

    /**
     * @Route("/receipts/{invoiceId}/print/{template_name}", name="print_receipt")
     */
    // public function printReceipt(Request $request, EntityManagerInterface $entityManager, Exporter $exporter, $invoiceId, $template_name)
    // {
    //     $invoiceRepository = $entityManager->getRepository(Facture::class);
    //     // $invoice = $invoiceRepository->createQueryBuilder('i')
    //     //     ->select('i.id', 'i.total', 'i.dateFact', 'i.typeFacture', 'c.nomClient')
    //     //     ->where('i.id = :invoiceId')
    //     //     ->innerJoin('i.client', 'c')
    //     //     ->leftJoin('i.licence', 'l')
    //     //     ->setParameter('invoiceId', $invoiceId)
    //     //     ->getQuery()
    //     //     ->getOneOrNullResult()
    //     // ;
    //     $invoice = $invoiceRepository->find($id);
    //     // dd($invoice);
    //     $printType = $request->query->get('printType');
    //     $fileName = \sprintf('%s %s %s', $printType, $invoice->getId(), $invoice->getClient()->getNomClient());
    //     $invoiceItemsCount = \count(\str_contains($invoice->getTypeFacture(), 'proforma') ? $invoice->getDetailsFactures() : $invoice->getPointOfSaleHistories());

    //     $company = $entityManager
    //         ->getRepository(Maison::class)
    //         ->createQueryBuilder('c')
    //         ->select('c.logo', 'c.nomSociete')
    //         ->getQuery()
    //         ->getSingleResult()
    //     ;

    //     $header = [
    //         'company' => $company,
    //         // 'vehicle' => [
    //         //     'genre' => $vehicle->getGenre(),
    //         //     'marque' => $vehicle->getMarque(),
    //         //     'typeOuModele' => $vehicle->getTypeOuModele(),
    //         //     'numeroImmatriculation' => $vehicle->getNumeroImmatriculation(),
    //         //     'kilometrage' => $vehicle->getKilometrage(),
    //         // ],
    //         'client' => $invoice->getClient()->getNomClient(),
    //     ];

    //     // $options['pageFooter'] = '{DATE F j, Y}|'.$fileName.'|Page {PAGENO} of {nb}';
    //     $options = [
    //         'header' => $header,
    //         // 'roll58' => [58, 1 * 6 + 18],
    //         // 'roll80' => [80, 1 * 6 + 18],
    //         'itemsContainer' => $request->query->get('itemsContainer')
    //     ];
    //     $staticContentHeight = 30;
    //     $lineHeight = 3;

    //     switch ($template_name) {
    //         case 'roll58':
    //             $options['config']['format'] = [58, $invoiceItemsCount * $lineHeight + $staticContentHeight];
    //             $options['config']['default_font_size'] = 8;
    //             $options['config']['default_font'] = 'monospace';
    //             break;
    //         case 'roll80':
    //             $options['config']['format'] = [80, $invoiceItemsCount * $lineHeight + $staticContentHeight];
    //             $options['config']['default_font_size'] = 9;
    //             $options['config']['default_font'] = 'monospace';
    //             break;
    //         default:
    //             $options['config']['format'] = $template_name;
    //             break;
    //     }
    //     // dd($options);
    //     $exporter->exportPdf(
    //         'export/receipt/'.$template_name.'.html.twig',
    //         $invoice,
    //         [],
    //         $fileName,
    //         $options
    //     );
    // }
}
