<?php

namespace App\Controller;

use App\AbstractController;
use App\Entity\Affectation;
use App\Entity\DestockingReason;
use App\Http\ApiResponse;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

class FicheStockController extends AbstractController
{
    /**
     * @Route("/fichestock", name="fiche_stock")
     */
    public function index(Request $request): Response
    {
        $entityManager = $this->getDoctrine()->getManager();
        $repository = $entityManager->getRepository($this->isGranted('ROLE_GESTSTOCK') ? 'App\Entity\DepotDetails' : 'App\Entity\PointOfSaleDetails');
        $dhr = $entityManager->getRepository($this->isGranted('ROLE_GESTSTOCK') ? 'App\Entity\DepotHistory' : 'App\Entity\PointOfSaleHistory');
        $strLimit = $request->get('limit');
        $search = $request->get('search');
        $limit = isset($strLimit) && is_string($strLimit) ? intval($strLimit) : 10;

        $queryBuilder = $repository
            ->createQueryBuilder('dd')
            ->select('dd.id', 'dd.quantity', 'dd.warningStockLevel', 'dd.idealStockLevel', 's.name AS product', 's.id as productId', 'u.name AS uomName', 'u.symbol AS uomSymbol', 'd.name AS depot', 'd.id AS depotId')
        ;

        if (isset($search)) {
            $queryBuilder = $queryBuilder->where($queryBuilder->expr()->like('s.name', ':search'));
        }

        if (!$this->isGranted('ROLE_GESTSTOCK')) {
            $pos = $entityManager->getRepository(Affectation::class)->findOneBy(['user' => $this->getUser()])->getPointOfSale();
            $queryBuilder->andWhere('p = :pos')->innerJoin('dd.pointOfSale', 'p');
        }

        $queryBuilder = $queryBuilder
            ->innerJoin('dd.service', 's')
            ->innerJoin($this->isGranted('ROLE_GESTSTOCK') ? 'dd.depot' : 'dd.pointOfSale', 'd')
            ->leftJoin('s.uom', 'u')
            ->orderBy('dd.createdAt', 'DESC')
        ;

        if (!$this->isGranted('ROLE_GESTSTOCK')) {
            $queryBuilder = $queryBuilder->setParameter('pos', $pos);
        }

        if (isset($search)) {
            $queryBuilder = $queryBuilder->setParameter('search', '%'.$search.'%');
        }

        $stocks = $queryBuilder
            ->setMaxResults($limit)
            ->getQuery()
            ->getResult()
        ;

        if ($request->isXmlHttpRequest()) {
            return new ApiResponse('', $stocks);
        }

        if ($request->isMethod('GET') && count($entityManager->getRepository(DestockingReason::class)->findAll()) === 0) {
            $defaults = [
                ['code' => 'peremption', 'name' => 'L\'article est périmé'],
                ['code' => 'casse', 'name' => 'Casse'],
                ['code' => 'consommation_interne', 'name' => 'Consommation interne'],
                ['code' => 'reajustement_stock', 'name' => 'Réajustement stock']
            ];

            foreach ($defaults as $default) {
                $motifDestockage = new DestockingReason();
                $motifDestockage->setCode($default['code']);
                $motifDestockage->setName($default['name']);
                $entityManager->persist($motifDestockage);
            }

            $entityManager->flush();
        }

        $qb = $dhr->createQueryBuilder('d')
            ->select('d.id as historyId', 's.name', 'd.expirationDate', 'd.quantity', 'd.releases', 'd.motifDestockage', 'd.note', 'dd.quantity as depotDetailQuantity', 'u.symbol', 'u.name as uomName', 'dp.id as depotId', 'dp.name as depotName')
            ->where('s.type = 2')
        ;

        if (!$this->isGranted('ROLE_GESTSTOCK')) {
            $pos = $entityManager->getRepository(Affectation::class)->findOneBy(['user' => $this->getUser()])->getPointOfSale();
            $qb->andWhere('p = :pos')->innerJoin('d.pos', 'p')->setParameter('pos', $pos);
        }

        $qb->innerJoin('d.service', 's')
            ->innerJoin($this->isGranted('ROLE_GESTSTOCK') ? 'd.depot' : 'd.pos', 'dp')
            ->innerJoin($this->isGranted('ROLE_GESTSTOCK') ? 's.depotDetails' : 's.pointOfSaleDetails', 'dd')
            ->leftJoin('s.uom', 'u')
        ;

        $rawServices = $qb->andWhere($qb->expr()->isNull('d.motifDestockage'))->getQuery()->getResult();
        $services = [];

        foreach ($rawServices as $service) {
            $services[] = [
                'historyId' => $service['historyId'],
                'depotId' => $service['depotId'],
                'depot' => $service['depotName'],
                'service' => $service['name'],
                'uom' => $service['symbol'],
                'uomName' => $service['uomName'],
                'quantity' => $service['quantity'],
                'motif' => $service['motifDestockage'],
                'note' => $service['note'],
                'label' => sprintf('%s, Entrée : %d, Disponible : %d %s%s', $service['name'], $service['quantity'], $service['depotDetailQuantity'], $service['symbol'], (!is_null($service['expirationDate']) ? ' (expire le '.$service['expirationDate']->format('d-m-Y').')' : ''))
            ];
        }

        if ($this->isGranted('ROLE_GESTSTOCK')) {
            $depots = array_map(function(array $depot): array {
                $depot['typeDepot'] = 'depot';
                return $depot;
            }, $entityManager->createQuery("SELECT d.id, d.name FROM App\Entity\Depot d ORDER BY d.name ASC")->getResult());

            $pointsOfSale = array_map(function(array $pos): array {
                $pos['typeDepot'] = 'pos';
                return $pos;
            }, $entityManager->createQuery("SELECT p.id, p.name FROM App\Entity\PointOfSale p ORDER BY p.name ASC")->getResult());

            $depots = array_merge($depots, $pointsOfSale);
        } else {
            $posId = $entityManager->getRepository(Affectation::class)->findOnePosIdByUser($this->getUser())['id'];
            $depots = $entityManager
                ->createQuery("SELECT d.id, d.name FROM App\Entity\PointOfSale d WHERE d.id = :posId")
                ->setParameter('posId', $posId)
                ->getResult()
            ;
        }

        $motifs = $entityManager
            ->createQuery("SELECT d.id, d.code, d.name AS label, d.description FROM App\Entity\DestockingReason d")
            ->getResult()
        ;

        $user = $this->getUser();
        $userArr = [
            'username' => $user->getUserIdentifier(),
            'fullName' => $user->getFullName(),
            'roles' => $user->getRoles()
        ];

        return $this->render('fichestock/index.html.twig', [
            'title' => 'Fiche de stock',
            'breadcrumb' => [
                ['name' => 'Dépôts', 'path' => '/depots'],
                ['name' => 'Fiche de stock']
            ],
            'services' => $services,
            'stocks' => $stocks,
            'depots' => $depots,
            'motifs' => $motifs,
            'user' => $userArr
        ]);
    }

    /**
     * @Route("/fichestock/{id}", name="fiche_stock_row")
     */
    public function ficheStock($id): JsonResponse
    {
        $data = json_decode(file_get_contents('php://input'), true);
        $em = $this->getDoctrine()->getManager();

        $er = $em->getRepository($this->isGranted('ROLE_GESTSTOCK') ? 'App\Entity\DepotDetails' : 'App\Entity\PointOfSaleDetails');;
        $entity = $er->find(intval($id));

        foreach ($data as $key => $value) {
            $setter = 'set'.ucfirst($key);
            $entity->$setter($value);
        }

        $em->flush();

        return new JsonResponse('Enregistré avec succès');
    }
}
