<?php

namespace App\Controller;

use App\AbstractController;
use App\Entity\UnitOfMeasure;
use App\Http\ApiResponse;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

class UnitOfMeasureController extends AbstractController
{
    /**
     * @Route("/uom", name="uom_list")
     */
    public function list(): Response
    {
        $entityRepository = $this->getDoctrine()->getRepository(UnitOfMeasure::class);

        return $this->render('unitsofmeasure/list.html.twig', [
            'title' => 'Unités de mesure',
            'breadcrumb' => [
                ['name' => 'Unités de mesure']
            ],
            'unitsofmeasure' => $entityRepository->findBy([], ['createdAt' => 'DESC']),
        ]);
    }

    /**
     * @Route("/uom/new", name="new_uom")
     * @IsGranted("ROLE_UOM", message="Vous n'êtes pas autorisé à ajouter une unité de mesure")
     */
    public function new(Request $request): Response
    {
        return $this->edit('unitsofmeasure/new.html.twig', $request);
    }

    /**
     * @Route("/uom/{id}", name="details_uom")
     */
    public function details(Request $request, int $id): Response
    {
        return $this->edit('unitsofmeasure/details.html.twig', $request, $id);
    }

    /**
     * @Route("/uom/{id}/delete", name="delete_uom")
     */
    public function delete(Request $request, int $id): Response
    {
        return $this->crud($request, $id, UnitOfMeasure::class, [], '', []);
    }

    private function edit(string $template, Request $request, $id = null): Response
    {
        $entityManager = $this->getDoctrine()->getManager();
        $entityRepository = $entityManager->getRepository(UnitOfMeasure::class);
        $uom = null;
        $violationList = [];

        if (isset($id)) {
            $uom = $entityRepository->find($id);
        }

        if ($request->getMethod() == 'POST') {
            if (!isset($id)) {
                $uom = new UnitOfMeasure();
            }
            $uom->setName($request->request->get('name'));
            if ($symbol = $request->request->get('symbol')) {
                $uom->setSymbol($symbol);
            }

            $violationList = $this->validator->validate($uom);
            $countViolations = count($violationList);
            $this->get('session')->getFlashBag()->clear();

            if ($countViolations > 0) {
                $message = 'Le formulaire contient des erreurs';
                if ($request->isXMLHttpRequest()) {
                    return new ApiResponse($message, [], $violationList, 500);
                }
                $this->addFlash('danger', $message);
            } else {
                if (!isset($id)) {
                    $entityManager->persist($uom);
                }
                $entityManager->flush();
                // do anything else you need here
                if (isset($id)) {
                    $placeholder = 'modifiée';
                } else {
                    $placeholder = 'ajoutée';
                }
                $this->addFlash('success', sprintf("Unité de mesure %s avec succès", $placeholder));
            }
        }
        
        return $this->render($template, [
            'title' => isset($id) ? 'Modifier unité de mesure' : 'Nouvelle unité de mesure',
            'breadcrumb' => [
                ['name' => 'Unités de mesure', 'path' => '/uom'],
                ['name' => isset($id) ? $uom->getName() : 'Nouvelle unité de mesure']
            ],
            'latestUnitsofmeasure' => $entityRepository->findBy([], ['createdAt' => 'DESC']),
            'currentU' => $uom,
            'violations' => $violationList
        ]);
    }
}
